/*
 * mcf_i2c.c
 *
 *  Created on: Mar 8, 2024
 *      Author: a0492361
 */
#include "ti_msp_dl_config.h"
#include "mcf_i2c.h"
#include "crc8.h"
#include "parity_even.h"

/*Counters for TX length and bytes sent*/
uint32_t gTxLen, gTxCount;

/*Data received from Target*/
uint8_t gRxPacket[I2C_RX_MAX_PACKET_SIZE];        // 4
uint8_t gRxPacketCRC[I2C_RX_MAX_PACKET_SIZECRC];  // 9
uint8_t gTxPacket[I2C_TX_MAX_PACKET_SIZE];        // 8
uint8_t gTxPacketCRC[I2C_TX_MAX_PACKET_SIZECRC];  // 9
bool parity_check;

/*Counters for RX length and bytes sent*/
uint32_t gRxLen, gRxCount;


bool I2C_write(unsigned long addr, unsigned long writedata) {

    /*Construct the 24 bit control word (refer to datasheet section 7.6.2.1)*/
    gTxCount = 0;
    uint8_t control_word[3] = {(uint8_t) DL_I2C_CONTROLLER_DIRECTION_TX << CONTROL_WORD_DIRECTION_OFFSET |
                               CRC_ENABLE << CONTROL_WORD_CRC_OFFSET |
                               DLEN_32 << CONTROL_WORD_DLEN_OFFSET | MEMORY_SECTION,
                               (MEMORY_PAGE << 4) | (addr & 0x00000F00)>>8, addr & 0x000000FF};

    /*Flushing the TX FIFO for any stray data*/
    //DL_I2C_flushControllerTXFIFO(I2C_INST);

    /*
     * Fill the FIFO with the control word and data to be sent
     * The FIFO is 8-bytes deep, and this function will return number
     * of bytes written to FIFO
     */
    gTxPacketCRC[0] = I2C_TARGET_ADDRESS<<1;
    for (int i = 0; i<3; i++){
        gTxPacket[i] =  control_word[i];
        gTxPacketCRC[i+1] =  gTxPacket[i];
    }
    for (int i = 0; i<4; i++){
        gTxPacket[6-i] =  writedata>>((3-i)*8)&0xFF;
        gTxPacketCRC[7-i] =  gTxPacket[6-i];
    }
    /*fill CRC bit*/
    gTxPacket[7] = crc8ccitt((uint8_t *) &gTxPacketCRC[0],8);
    gTxPacketCRC[8] = gTxPacket[7];

    gTxCount += DL_I2C_fillControllerTXFIFO(I2C_INST, (uint8_t *) &gTxPacket[0] , I2C_TX_MAX_PACKET_SIZE);

    /*Enable TXFIFO trigger interrupt if there are more bytes to send*/
    if (gTxCount < I2C_TX_MAX_PACKET_SIZE) {
        DL_I2C_enableInterrupt(I2C_INST, DL_I2C_INTERRUPT_CONTROLLER_TXFIFO_TRIGGER);
    }
    else {
        DL_I2C_disableInterrupt(I2C_INST, DL_I2C_INTERRUPT_CONTROLLER_TXFIFO_TRIGGER);
    }

    gI2cControllerStatus = I2C_STATUS_TX_STARTED;

    /*Check if I2C line is idle*/
    while (!(DL_I2C_getControllerStatus(I2C_INST) & DL_I2C_CONTROLLER_STATUS_IDLE));

    /* Send the packet to the controller.
     * This function will send Start + Stop automatically.
     */
    DL_I2C_startControllerTransfer(I2C_INST, I2C_TARGET_ADDRESS, DL_I2C_CONTROLLER_DIRECTION_TX,
                                  I2C_TX_MAX_PACKET_SIZE);

    /*Wait until the Controller sends all bytes*/
    while ((gI2cControllerStatus != I2C_STATUS_TX_COMPLETE) &&
            (gI2cControllerStatus != I2C_STATUS_ERROR)) {
        __WFE();
    }

    //while (DL_I2C_getControllerStatus(I2C_INST) & DL_I2C_CONTROLLER_STATUS_BUSY_BUS);

    /*Trap if there was an error*/
    if (DL_I2C_getControllerStatus(I2C_INST) & DL_I2C_CONTROLLER_STATUS_ERROR) {
        //__BKPT(0);
        return 0;
    }

    while (!(DL_I2C_getControllerStatus(I2C_INST) & DL_I2C_CONTROLLER_STATUS_IDLE));

    /*Add delay between transfers*/
    delay_cycles(3200);
    return 1;
}


bool I2C_read(unsigned long addr, unsigned long *result) {

    gRxCount = 0;

    /*Flush the FIFO to clear any stray data*/
    //DL_I2C_flushControllerRXFIFO(I2C_INST);
    //DL_I2C_flushControllerTXFIFO(I2C_INST);

    /*Construct the 24 bit control word (refer to datasheet section 7.6.2.1)*/
    uint8_t control_word[3] = {DL_I2C_CONTROLLER_DIRECTION_RX << CONTROL_WORD_DIRECTION_OFFSET |
                               CRC_ENABLE << CONTROL_WORD_CRC_OFFSET | DLEN_32 << CONTROL_WORD_DLEN_OFFSET |
                               MEMORY_SECTION , (MEMORY_PAGE << 4) | (addr & 0x00000F00)>>8 ,
                               addr & 0x000000FF};

    /*Fill the TX FIFO with the control word*/
    gTxCount = DL_I2C_fillControllerTXFIFO(I2C_INST, &control_word[0], CONTROL_WORD_LENGTH);

    while (!(DL_I2C_getControllerStatus(I2C_INST) & DL_I2C_CONTROLLER_STATUS_IDLE));

    gI2cControllerStatus = I2C_STATUS_RX_STARTED;

    /*Update the I2C Target Address in the MSA register with the Read bit set*/
    DL_Common_updateReg(&I2C_INST->MASTER.MSA,
                ((I2C_TARGET_ADDRESS << I2C_MSA_SADDR_OFS) | (uint32_t) DL_I2C_CONTROLLER_DIRECTION_RX),
                (I2C_MSA_SADDR_MASK | I2C_MSA_DIR_MASK));

    /*Enable RD_ON_TXEMPTY to start receiving data after sending the control word without a stop condition*/
    DL_Common_updateReg(&I2C_INST->MASTER.MCTR,(((uint32_t) gRxLen << I2C_MCTR_MBLEN_OFS) | I2C_MCTR_BURSTRUN_ENABLE |
            I2C_MCTR_RD_ON_TXEMPTY_ENABLE | DL_I2C_CONTROLLER_START_ENABLE | DL_I2C_CONTROLLER_STOP_ENABLE |
            DL_I2C_CONTROLLER_ACK_DISABLE), (I2C_MCTR_MBLEN_MASK | I2C_MCTR_RD_ON_TXEMPTY_MASK  |
                    I2C_MCTR_BURSTRUN_MASK | I2C_MCTR_START_MASK | I2C_MCTR_STOP_MASK | I2C_MCTR_ACK_MASK));

    /*Wait for all bytes to be received in interrupt*/
    while (gI2cControllerStatus != I2C_STATUS_RX_COMPLETE) {
        __WFE();
    }

    DL_I2C_disableControllerReadOnTXEmpty(I2C_INST);

    while (DL_I2C_getControllerStatus(I2C_INST) & DL_I2C_CONTROLLER_STATUS_BUSY_BUS);

    /*Construct the final 4 byte data read from the gRxPacket array*/
    *result = (unsigned long) gRxPacket[0] | ((unsigned long) gRxPacket[1] << 8) |
            ((unsigned long) gRxPacket[2] << 16) | ((unsigned long) gRxPacket[3] << 24);

    gRxPacketCRC[0] = I2C_TARGET_ADDRESS<<1;
    for (int i = 0; i<3; i++){
        gRxPacketCRC[i+1] =  control_word[i];
      }
    gRxPacketCRC[4] =  0x03;
      for (int i = 0; i<4; i++){
          gRxPacketCRC[5+i] =  gRxPacket[i];
      }
    /*fill CRC bit*/
    gRxPacketCRC[9] = crc8ccitt((uint8_t *) &gRxPacketCRC[0], 9);

    delay_cycles(3200);

    parity_check = parity_cal(*result,32);

    return 1;
}


